"use client";

import { useState } from "react";
import Button from "../common/Button/Button";
import { toast, ToastContainer } from "react-toastify";

const IntakeForm = () => {
  const [formData, setFormData] = useState({
    fullName: "",
    company: "",
    email: "",
    phone: "",
    contactMethod: [],
    hasWebsite: "",
    websiteURL: "",
    purpose: [],
    businessDescription: "",
    websiteGoals: "",
    features: [],
    hasLogo: "",
    needLogoHelp: "",
    websitesLiked: "",
    designPreferences: "",
    servicesNeeded: [],
    desiredLaunchDate: "",
    budget: "",
    additionalNotes: "",
  });

  const handleChange = (e) => {
    const { name, value, type, checked } = e.target;

    if (type === "checkbox") {
      setFormData((prev) => {
        const current = prev[name] || [];
        return {
          ...prev,
          [name]: checked
            ? [...current, value]
            : current.filter((item) => item !== value),
        };
      });
    } else {
      setFormData((prev) => ({ ...prev, [name]: value }));
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      const res = await fetch("/api/sendMail", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(formData),
      });
      const result = await res.json();
      if (result.success) {
        toast.success("Form sent successfully!");
        setFormData({
          fullName: "",
          company: "",
          email: "",
          phone: "",
          contactMethod: [],
          hasWebsite: "",
          websiteURL: "",
          purpose: [],
          businessDescription: "",
          websiteGoals: "",
          features: [],
          hasLogo: "",
          needLogoHelp: "",
          websitesLiked: "",
          designPreferences: "",
          servicesNeeded: [],
          desiredLaunchDate: "",
          budget: "",
          additionalNotes: "",
        });
      } else {
        toast.error("Error sending form!");
      }
    } catch (err) {
      toast.error("Server error!");
    }
  };
  return (
    <div>
      <h2 className="text-2xl uppercase font-bold mb-6 text-[#717175]">
        Website Project Intake Form
      </h2>
      <form onSubmit={handleSubmit} className="space-y-8 text-sm">
        {/* Client Information */}
        <div>
          <h3 className="text-lg uppercase text-[#161617] font-semibold mb-4">
            Client Information
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <input
              type="text"
              name="fullName"
              placeholder="Full Name"
              value={formData.fullName}
              onChange={handleChange}
              className="bg-[#f2f2f6] px-7 py-5 focus:outline-none focus:ring-2 focus:ring-[#0480b4] w-full"
            />
            <input
              type="text"
              name="company"
              placeholder="Company/Organization Name"
              value={formData.company}
              onChange={handleChange}
              className="bg-[#f2f2f6] px-7 py-5 focus:outline-none focus:ring-2 focus:ring-[#0480b4] w-full"
            />
          </div>

          {/* Email + Phone */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
            <input
              type="email"
              name="email"
              placeholder="Email Address"
              value={formData.email}
              onChange={handleChange}
              className="bg-[#f2f2f6] px-7 py-5 focus:outline-none focus:ring-2 focus:ring-[#0480b4] w-full"
            />
            <input
              type="tel"
              name="phone"
              placeholder="Phone Number"
              value={formData.phone}
              onChange={handleChange}
              className="bg-[#f2f2f6] px-7 py-5 focus:outline-none focus:ring-2 focus:ring-[#0480b4] w-full"
            />
          </div>
          <div className="mt-4">
            <label className="block font-medium mb-2">
              Preferred Contact Method:
            </label>
            {["Email", "Phone", "WhatsApp", "Other"].map((method) => (
              <label key={method} className="mr-4">
                <input
                  type="checkbox"
                  name="contactMethod"
                  value={method}
                  checked={formData.contactMethod.includes(method)}
                  onChange={handleChange}
                  className="mr-1"
                />
                {method}
              </label>
            ))}
          </div>
        </div>

        <div className="w-full h-0.5 bg-[#0480b4]" />

        {/* Project Overview */}
        <div>
          <h3 className="text-lg uppercase font-semibold mb-4">
            Project Overview
          </h3>
          <label className="block mb-2">Do you currently have a website?</label>
          <label className="mr-4">
            <input
              type="radio"
              name="hasWebsite"
              value="Yes"
              checked={formData.hasWebsite === "Yes"}
              onChange={handleChange}
              className="mr-1"
            />
            Yes
          </label>
          <label>
            <input
              type="radio"
              name="hasWebsite"
              value="No"
              checked={formData.hasWebsite === "No"}
              onChange={handleChange}
              className="mr-1"
            />
            No
          </label>
          {formData.hasWebsite === "Yes" && (
            <input
              type="text"
              name="websiteURL"
              placeholder="Website URL"
              value={formData.websiteURL}
              onChange={handleChange}
              className="w-full mt-3 p-3 bg-gray-100"
            />
          )}

          <label className="block mt-4 mb-2">
            What is the primary purpose of the new website?
          </label>
          {["Informational", "E-commerce", "Portfolio", "Blog", "Other"].map(
            (purpose) => (
              <label key={purpose} className="block">
                <input
                  type="checkbox"
                  name="purpose"
                  value={purpose}
                  checked={formData.purpose.includes(purpose)}
                  onChange={handleChange}
                  className="mr-2 mb-3"
                />
                {purpose}
              </label>
            )
          )}

          <textarea
            name="businessDescription"
            placeholder="Describe your business or organization briefly"
            value={formData.businessDescription}
            onChange={handleChange}
            className="w-full mt-4 p-3 bg-gray-100"
            rows={3}
            maxLength={500}
          ></textarea>

          <textarea
            name="websiteGoals"
            placeholder="What goals do you want to achieve with this website?"
            value={formData.websiteGoals}
            onChange={handleChange}
            className="w-full mt-4 p-3 bg-gray-100"
            rows={3}
            maxLength={500}
          ></textarea>
        </div>

        <div className="w-full h-0.5 bg-[#0480b4]" />

        {/* Features */}
        <div>
          <h3 className="text-lg uppercase font-semibold mb-4">
            Website Features You Need
          </h3>
          {[
            "Contact Form",
            "Blog",
            "Booking System",
            "Online Store",
            "Payment Integration",
            "SEO Optimization",
            "Responsive/Mobile Design",
            "Social Media Integration",
            "Newsletter Signup",
            "Analytics/Tracking",
            "Other",
          ].map((feature) => (
            <label key={feature} className="block">
              <input
                type="checkbox"
                name="features"
                value={feature}
                checked={formData.features.includes(feature)}
                onChange={handleChange}
                className="mr-2 mb-3"
              />
              {feature}
            </label>
          ))}
        </div>

        <div className="w-full h-0.5 bg-[#0480b4]" />

        {/* Design Preferences */}
        <div>
          <h3 className="text-lg uppercase font-semibold mb-4">
            Design Preferences
          </h3>
          <label className="block">
            Do you have a logo and brand materials?
          </label>
          <label className="mr-4">
            <input
              type="radio"
              name="hasLogo"
              value="Yes"
              checked={formData.hasLogo === "Yes"}
              onChange={handleChange}
              className="mr-1 mt-3"
            />
            Yes
          </label>
          <label>
            <input
              type="radio"
              name="hasLogo"
              value="No"
              checked={formData.hasLogo === "No"}
              onChange={handleChange}
              className="mr-1 mt-3"
            />
            No
          </label>

          {formData.hasLogo === "No" && (
            <div className="mt-2">
              <h2>Do you need a logo?</h2>
              <label className="mr-4">
                <input
                  type="radio"
                  name="needLogoHelp"
                  value="Yes"
                  checked={formData.needLogoHelp === "Yes"}
                  onChange={handleChange}
                  className="mr-1 mt-3"
                />
                Yes
              </label>
              <label>
                <input
                  type="radio"
                  name="needLogoHelp"
                  value="No"
                  checked={formData.needLogoHelp === "No"}
                  onChange={handleChange}
                  className="mr-1 mt-3"
                />
                No
              </label>
            </div>
          )}

          <textarea
            name="websitesLiked"
            placeholder="Any websites you like the look of?"
            value={formData.websitesLiked}
            onChange={handleChange}
            className="w-full mt-4 p-3 bg-gray-100"
            rows={2}
            maxLength={500}
          ></textarea>

          <textarea
            name="designPreferences"
            placeholder="Preferred colors, style, or design notes"
            value={formData.designPreferences}
            onChange={handleChange}
            className="w-full mt-4 p-3 bg-gray-100"
            rows={2}
            maxLength={500}
          ></textarea>
        </div>

        <div className="w-full h-0.5 bg-[#0480b4]" />

        {/* Services Needed */}
        <div>
          <h3 className="text-lg uppercase font-semibold mb-4">
            Web Skills or Services Needed
          </h3>
          {[
            "Web Design",
            "Front-End Development",
            "Back-End Development",
            "WordPress or CMS setup",
            "E-commerce Platform Setup",
            "SEO & Digital Marketing",
            "Ongoing Maintenance & Updates",
            "Content Creation",
            "Hosting & Domain",
          ].map((service) => (
            <label key={service} className="block">
              <input
                type="checkbox"
                name="servicesNeeded"
                value={service}
                checked={formData.servicesNeeded.includes(service)}
                onChange={handleChange}
                className="mr-2 mb-3"
              />
              {service}
            </label>
          ))}
        </div>
        <div className="w-full h-0.5 bg-[#0480b4]" />

        {/* Timeline and Budget */}
        <div>
          <label className="block uppercase font-semibold mb-2">
            Desired Launch Date
          </label>
          <input
            name="desiredLaunchDate"
            value={formData.timeline}
            onChange={handleChange}
            placeholder="Example: 2 months, 3-4 weeks, ASAP..."
            className="bg-[#f2f2f6] px-7 py-5 focus:outline-none focus:ring-2 focus:ring-[#0480b4] w-full"
          />
        </div>

        <div className="space-y-2">
          <label className="block uppercase font-medium">
            Estimated Budget
          </label>
          {[
            "<$1,000",
            "$1,000–$3,000",
            "$3,000–$5,000",
            "$5,000+",
            "Not sure yet",
          ].map((option) => (
            <label key={option} className="flex items-center gap-2">
              <input
                type="radio"
                name="budget"
                value={option}
                checked={formData.budget === option}
                onChange={handleChange}
                className="w-5 h-5 text-[#0480b4] focus:ring-[#0480b4]"
              />
              {option}
            </label>
          ))}
        </div>
        <div className="w-full h-0.5 bg-[#0480b4]" />

        {/* Additional Notes */}
        <div>
          <label className="block uppercase font-semibold mb-2">
            Additional Notes
          </label>
          <textarea
            name="additionalNotes"
            value={formData.additionalNotes}
            onChange={handleChange}
            placeholder="Any extra details or special requirements..."
            className="bg-[#f2f2f6] px-7 py-5 focus:outline-none focus:ring-2 focus:ring-[#0480b4] w-full"
            maxLength={500}
          ></textarea>
        </div>

        {/* Submit */}
        <div className="flex justify-center">
          <Button text="Submit Form" isForm={true} />
        </div>
      </form>
      <ToastContainer position="top-right" autoClose={4000} />
    </div>
  );
};

export default IntakeForm;
